package net.danygames2014.nyalib.network.energy;

import net.danygames2014.nyalib.energy.EnergyConsumer;
import net.danygames2014.nyalib.energy.EnergySource;
import net.danygames2014.nyalib.network.*;
import net.minecraft.class_18;
import net.minecraft.class_63;
import net.modificationstation.stationapi.api.util.math.Direction;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

public class EnergyNetwork extends Network {
    public HashMap<class_63, ConsumerEntry> consumers;

    public HashMap<class_63, ArrayList<ConsumerPath>> consumerCache;

    public EnergyNetwork(class_18 world, NetworkType type) {
        super(world, type);
        consumers = new HashMap<>();
        consumerCache = new HashMap<>();
    }

    @Override
    public void update() {
        super.update();

        consumers.clear();
        consumerCache.clear();
        for (NetworkComponentEntry componentEntry : components.values()) {
            if (componentEntry.component() instanceof NetworkEdgeComponent) {
                if (world.method_1777(componentEntry.pos().field_1482, componentEntry.pos().field_1483, componentEntry.pos().field_1484) instanceof EnergyConsumer consumer) {
                    consumers.put(componentEntry.pos(), new ConsumerEntry(componentEntry, consumer));
                }
            }
        }
    }

    //long time = System.nanoTime();
    //System.out.println(((System.nanoTime() - time) / 1000) + "uS");

    /**
     * Provide energy to the energy net
     *
     * @param source    The source of energy
     * @param sourcePos The position of the source
     * @param voltage   The voltage provided
     * @param power  The amperage provided
     * @return The power used
     */
    public int provideEnergy(EnergySource source, class_63 sourcePos, int voltage, int power) {
        int remainingPower = power;

        for (ConsumerPath consumerPath : getValidConsumers(sourcePos)) {
            EnergyConsumer consumer = consumerPath.consumer;
            NetworkPath path = consumerPath.path;

            // Check if the consumer can even accept more energy
            if (consumer.getRemainingCapacity() > 0) {
                // Insert Energy into the consumers
                int usedPower = traverseEnergy(consumer, path.endFace, voltage, remainingPower);

                // Reduce the remaining amount
                remainingPower -= usedPower;

                // If there are 0 amps remaining, end it
                if (remainingPower <= 0) {
                    return power;
                }
            }

        }

        return power - remainingPower;
    }
    
    private int traverseEnergy(EnergyConsumer consumer, Direction consumerFace, int voltage, int remainingPower){
        return consumer.receiveEnergy(consumerFace, voltage, remainingPower);
    }

    /**
     * Retrieves all the valid reachable consumers from the source position
     *
     * @param source The position of the source
     * @return An ArrayList of paths to valid consumers
     */
    public ArrayList<ConsumerPath> getValidConsumers(class_63 source) {
        if (consumerCache.containsKey(source)) {
            return consumerCache.get(source);
        } else {
            ArrayList<ConsumerPath> consumers = new ArrayList<>();

            for (Map.Entry<class_63, ConsumerEntry> consumer : this.consumers.entrySet()) {
                if (consumer.getKey().equals(source)) {
                    continue;
                }

                NetworkPath path = this.getPath(source, consumer.getKey());
                if (consumer.getValue().consumer.canReceiveEnergy(path.endFace)) {
                    consumers.add(new ConsumerPath(consumer.getValue().consumer, path));
                }
            }

            consumerCache.put(source, consumers);
            return consumers;
        }
    }

    public record ConsumerPath(EnergyConsumer consumer, NetworkPath path) {
    }

    public record ConsumerEntry(NetworkComponentEntry componentEntry, EnergyConsumer consumer) {
    }
}
