package net.danygames2014.nyalib.item;

import net.minecraft.class_18;
import net.minecraft.class_31;
import net.modificationstation.stationapi.api.util.math.Direction;
import org.jetbrains.annotations.Nullable;

@SuppressWarnings("unused")
public interface ItemHandler {
    /**
     * Check if the block supports extracting items on this side, if this returns false there
     * should be no point in trying to use {@link #extractItem(int, int, Direction)}
     *
     * @param direction Direction to check
     * @return <code>true</code> if the device supports item extraction from the given direction
     */
    boolean canExtractItem(@Nullable Direction direction);

    /**
     * Extract an item from the block
     *
     * @param slot      The slot to extract from
     * @param amount    The amount to extract (can be larger than the maximum stack size)
     * @param direction The direction to extract from
     * @return The ItemStack extracted, null if nothing is extracted
     */
    class_31 extractItem(int slot, int amount, @Nullable Direction direction);

    /**
     * Check if the block supports inserting items on this side, if this returns false there
     * should be no point in trying to use {@link #insertItem(class_31, Direction)} or {@link #insertItem(class_31, int, Direction)}
     *
     * @param direction Direction to check
     * @return <code>true</code> if the device supports item insertion from the given direction
     */
    boolean canInsertItem(@Nullable Direction direction);

    /**
     * Insert item into the given slot and return the remainder
     *
     * @param stack     The {@link class_31} to insert
     * @param slot      Slot to insert into
     * @param direction Direction to insert from
     * @return The remainder of the ItemStack (null if it was inserted entirely), this should be a new ItemStack, however it can be the same if it was not modified
     */
    class_31 insertItem(class_31 stack, int slot, @Nullable Direction direction);

    /**
     * Insert item into any slot and return the remainder
     *
     * @param stack     The {@link class_31} to insert
     * @param direction Direction to insert from
     * @return The remainder of the ItemStack (null if it was inserted entirely), this should be a new ItemStack, however it can be the same if it was not modified
     */
    class_31 insertItem(class_31 stack, @Nullable Direction direction);

    /**
     * Get the {@link class_31} in the given slot, If there is no {@link class_31}, then return null
     * <p>
     *
     * @param slot      The slot to get the {@link class_31} from
     * @param direction The direction to query from
     * @return The {@link class_31} in the slot
     */
    class_31 getStackInSlot(int slot, @Nullable Direction direction);

    /**
     * Get the size of the block inventory
     *
     * @return The number of slots this block has
     */
    int getSize(@Nullable Direction direction);

    /**
     * Attempts to send {@link class_31} to the given side
     *
     * @param world     The world this device is in
     * @param x         The x-position of this device
     * @param y         The y-position of this device
     * @param z         The z-position of this device
     * @param direction The direction you want to send power in
     * @return The remainder of the {@link class_31} (null if it was inserted entirely). If there is no neighbor in that direction, returns the same {@link class_31}
     */
    default class_31 sendItem(class_18 world, int x, int y, int z, class_31 stack, Direction direction) {
        ItemHandler neighbor = getNeighborItemHandler(world, x, y, z, direction);

        if (neighbor == null) {
            return stack;
        }

        return neighbor.insertItem(stack, direction.getOpposite());
    }

    /**
     * Attempts to send {@link class_31} to the given side into a given slot
     *
     * @param world     The world this device is in
     * @param x         The x-position of this device
     * @param y         The y-position of this device
     * @param z         The z-position of this device
     * @param direction The direction you want to send power in
     * @return The remainder of the {@link class_31} (null if it was inserted entirely). If there is no neighbor in that direction, returns the same {@link class_31}
     */
    default class_31 sendItem(class_18 world, int x, int y, int z, class_31 stack, int slot, Direction direction) {
        ItemHandler neighbor = getNeighborItemHandler(world, x, y, z, direction);

        if (neighbor == null) {
            return stack;
        }

        return neighbor.insertItem(stack, slot, direction.getOpposite());
    }

    /**
     * Attempts to get a neighboring {@link ItemHandler}
     *
     * @param world     The world this device is in
     * @param x         The x-position of this device
     * @param y         The y-position of this device
     * @param z         The z-position of this device
     * @param direction The direction you want to look for the neighbor in
     * @return The neighbor's {@link ItemHandler}, if there is not a neighboring {@link ItemHandler} then returns <code>null</code>
     */
    default ItemHandler getNeighborItemHandler(class_18 world, int x, int y, int z, Direction direction) {
        if (direction == null) {
            return null;
        }

        if (world.method_1777(x + direction.getOffsetX(), y + direction.getOffsetY(), z + direction.getOffsetZ()) instanceof ItemHandler handler) {
            return handler;
        }

        return null;
    }
}
