package net.danygames2014.nyalib.network;

import java.util.ArrayList;
import net.minecraft.class_17;
import net.minecraft.class_18;
import net.minecraft.class_8;

@SuppressWarnings("unused")
public interface NetworkComponent {
    /**
     * This is a helper method if you only need one network.
     *
     * @return A network type this component can participate in
     */
    NetworkType getNetworkType();

    /**
     * @return A list of network types this component can participate in
     */
    default ArrayList<NetworkType> getNetworkTypes() {
        ArrayList<NetworkType> types = new ArrayList<>();
        types.add(getNetworkType());
        return types;
    }

    /**
     * Called when the physical topology of the network updates
     *
     * @param world   The world this network is in
     * @param x       The x-position of this component
     * @param y       The y-position of this component
     * @param z       The z-position of this component
     * @param network The network this component is in
     */
    default void update(class_18 world, int x, int y, int z, Network network) {

    }

    /**
     * Called when this component is added to the network
     *
     * @param world   The world this network is in
     * @param x       The x-position of this component
     * @param y       The y-position of this component
     * @param z       The z-position of this component
     * @param network The network this component is in
     */
    default void onAddedToNet(class_18 world, int x, int y, int z, Network network) {

    }

    /**
     * Called when this component is removed from the network
     * (it's called right before it is removed from the network)
     *
     * @param world   The world this network is in
     * @param x       The x-position of this component
     * @param y       The y-position of this component
     * @param z       The z-position of this component
     * @param network The network this component is in
     */
    default void onRemovedFromNet(class_18 world, int x, int y, int z, Network network) {

    }

    /**
     * Called when this component is automatically added to network
     *
     * @param world     The world this component is in
     * @param x         The x-position of this component
     * @param y         The y-position of this component
     * @param z         The z-position of this component
     * @param component Reference to this component
     * @param <T>       {@link class_17} implementing a {@link NetworkComponent}
     */
    default <T extends class_17 & NetworkComponent> void addToNet(class_18 world, int x, int y, int z, T component) {
        NetworkManager.addBlock(world, x, y, z, component);
    }

    /**
     * Called when this component is automatically removed from network
     *
     * @param world     The world this component is in
     * @param x         The x-position of this component
     * @param y         The y-position of this component
     * @param z         The z-position of this component
     * @param component Reference to this component
     * @param <T>       {@link class_17} implementing a {@link NetworkComponent}
     */
    default <T extends class_17 & NetworkComponent> void removeFromNet(class_18 world, int x, int y, int z, T component) {
        NetworkManager.removeBlock(world, x, y, z, component);
    }

    /**
     * Called when this network is being saved
     *
     * @param world   The world this network is in
     * @param x       The x-position of this component
     * @param y       The y-position of this component
     * @param z       The z-position of this component
     * @param network The network this component is in
     * @param nbt     The NBT Compound being saved
     */
    default void writeNbt(class_18 world, int x, int y, int z, Network network, class_8 nbt) {

    }

    /**
     * Called when this network is being loaded
     *
     * @param world   The world this network is in
     * @param x       The x-position of this component
     * @param y       The y-position of this component
     * @param z       The z-position of this component
     * @param network The network this component is in
     * @param nbt     The NBT Compound being read
     */
    default void readNbt(class_18 world, int x, int y, int z, Network network, class_8 nbt) {

    }
}
