package net.danygames2014.nyalib.network;

import net.danygames2014.nyalib.NyaLib;
import net.minecraft.class_17;
import net.minecraft.class_18;
import net.minecraft.class_50;
import net.minecraft.class_63;
import net.minecraft.class_8;
import net.modificationstation.stationapi.api.registry.DimensionRegistry;
import net.modificationstation.stationapi.api.util.Identifier;
import net.modificationstation.stationapi.api.util.math.Direction;

import java.util.*;
import java.util.Map.Entry;
import java.util.concurrent.atomic.AtomicInteger;

@SuppressWarnings({"UnusedReturnValue", "DuplicatedCode", "LoggingSimilarMessage", "CollectionAddAllCanBeReplacedWithConstructor"})
public class NetworkManager {
    /**
     * For each Dimension there is a hashmap which takes network type Identifier as a key
     */
    public static HashMap<class_50, HashMap<Identifier, ArrayList<Network>>> NETWORKS = new HashMap<>();

    public static AtomicInteger NEXT_ID = new AtomicInteger(0);

    public static ArrayList<Network> removeQueue = new ArrayList<>();

    // Getting Networks
    public static ArrayList<Network> getNetworks(class_50 dimension, Identifier networkTypeIdentifier) {
        HashMap<Identifier, ArrayList<Network>> netDims = NETWORKS.get(dimension);

        if (netDims == null) {
            return new ArrayList<>();
        }

        ArrayList<Network> nets = netDims.get(networkTypeIdentifier);
        return nets != null ? nets : new ArrayList<>();
    }

    public static HashMap<Identifier, ArrayList<Network>> getNetworks(class_50 dimension) {
        return NETWORKS.get(dimension);
    }

    // Adding a network
    @SuppressWarnings("Java8MapApi")
    public static void addNetwork(class_50 dimension, Network network) {
        HashMap<Identifier, ArrayList<Network>> dimNetworks = NETWORKS.get(dimension);

        if (network == null) {
            return;
        }

        if (dimNetworks == null) {
            dimNetworks = new HashMap<>();
            NETWORKS.put(dimension, dimNetworks);
        }

        ArrayList<Network> typeNetworks = dimNetworks.get(network.type.getIdentifier());
        if (typeNetworks == null) {
            typeNetworks = new ArrayList<>();
            dimNetworks.put(network.type.getIdentifier(), typeNetworks);
        }

        typeNetworks.add(network);
    }

    public static Network createNetwork(class_50 dimension, NetworkType networkType) {
        Network network;

        try {
            network = networkType.getNetworkClass().getDeclaredConstructor(class_18.class, NetworkType.class).newInstance(dimension.field_2173, networkType);
        } catch (Exception e) {
            NyaLib.LOGGER.error("Error when creating a network of type {}", networkType.getIdentifier(), e);
            return null;
        }

        network.id = NEXT_ID.getAndIncrement();
        addNetwork(dimension, network);
        return network;
    }

    public static void removeNetwork(Network network) {
        if (!removeQueue.contains(network)) {
            removeQueue.add(network);
        }
    }

    public static void removeQueuedNetworks() {
        for (Network toremove : removeQueue) {
            removeNetworkInternal(toremove);
        }
    }

    private static boolean removeNetworkInternal(Network toRemove) {
        for (var dimensions : NETWORKS.entrySet()) {
            for (var networks : dimensions.getValue().entrySet()) {
                for (var network : networks.getValue()) {
                    if (network == toRemove) {
                        networks.getValue().remove(network);
                        return true;
                    }
                }
            }
        }
        return false;
    }

    /**
     * Get a network of the given type at the given coords
     *
     * @param dimension             The dimension to check in
     * @param x                     x-position to check
     * @param y                     y-position to check
     * @param z                     z-position to checj
     * @param networkTypeIdentifier The type of network to check for
     * @return A network if it exists on the coordinates. null if there is no network on the given coordinates.
     */
    public static Network getAt(class_50 dimension, int x, int y, int z, Identifier networkTypeIdentifier) {
        for (Network net : getNetworks(dimension, networkTypeIdentifier)) {
            if (net.isAt(x, y, z)) {
                return net;
            }
        }
        return null;
    }

    /**
     * Returns an ArrayList of networks of any type neighboring this block
     *
     * @param world The world to check in
     * @param x     x-position to check
     * @param y     y-position to check
     * @param z     z-position to checj
     * @return An ArrayList of networks neighboring this block
     */
    public static ArrayList<Network> getNeighbors(class_18 world, int x, int y, int z) {
        ArrayList<Network> neighborNets = new ArrayList<>();

        for (var networksOfType : getNetworks(world.field_216).values()) {
            for (var network : networksOfType) {
                for (Direction direction : Direction.values()) {
                    if (network.isAt(x + direction.getOffsetX(), y + direction.getOffsetY(), z + direction.getOffsetZ())) {
                        neighborNets.add(network);
                    }
                }
            }
        }

        return neighborNets;
    }

    /**
     * Returns an ArrayList of networks of a given type neighboring this block
     *
     * @param world                 The world to check in
     * @param x                     x-position to check
     * @param y                     y-position to check
     * @param z                     z-position to checj
     * @param networkTypeIdentifier The type of network to check for
     * @return An ArrayList of networks neighboring this block
     */
    public static ArrayList<Network> getNeighbors(class_18 world, int x, int y, int z, Identifier networkTypeIdentifier) {
        ArrayList<Network> neighborNets = new ArrayList<>();

        for (Network network : getNetworks(world.field_216, networkTypeIdentifier)) {
            for (Direction direction : Direction.values()) {
                if (network.isAt(x + direction.getOffsetX(), y + direction.getOffsetY(), z + direction.getOffsetZ())) {
                    neighborNets.add(network);
                }
            }
        }

        return neighborNets;
    }

    // Adding and Removing Blocks
    @SuppressWarnings("RedundantLabeledSwitchRuleCodeBlock")
    public static <T extends class_17 & NetworkComponent> void addBlock(class_18 world, int x, int y, int z, T component) {
        if (component == null) {
            return;
        }

        // For each of the network types this network component can handle, discover and add to networks
        for (NetworkType networkType : component.getNetworkTypes()) {
            ArrayList<Network> neighborNets = new ArrayList<>(2);
            ArrayList<Network> potentialNets = new ArrayList<>();

            potentialNets.addAll(getNetworks(world.field_216, networkType.getIdentifier()));

//            System.out.println("FOUND " + potentialNets.size() + " POTENTIAL NETWORKS FOR " + typeIdentifier);

            for (Network potentialNet : potentialNets) {
                for (Direction direction : Direction.values()) {
                    if (potentialNet.isAt(x + direction.getOffsetX(), y + direction.getOffsetY(), z + direction.getOffsetZ())) {
                        neighborNets.add(potentialNet);
                    }
                }
            }

//            System.out.println("FOUND " + neighborNets.size() + " NEIGHBOR NETWORKS FOR " + typeIdentifier);

            Network network;

            switch (neighborNets.size()) {
                case 0 -> {
                    network = createNetwork(world.field_216, networkType);
                }

                case 1 -> {
                    network = neighborNets.get(0);
                }

                default -> {
                    network = neighborNets.get(0);
                    for (int i = 1; i < neighborNets.size(); i++) {

                        if (neighborNets.get(i).getId() == network.getId()) {
                            continue;
                        }

                        network.blocks.putAll(neighborNets.get(i).blocks);
                        neighborNets.get(i).blocks.clear();
                        removeNetwork(neighborNets.get(i));
                    }
                }
            }

            if (network != null) {
                network.addBlock(x, y, z, component);
                network.update();
            }
        }
    }

    @SuppressWarnings("RedundantLabeledSwitchRuleCodeBlock")
    public static <T extends class_17 & NetworkComponent> void removeBlock(class_18 world, int x, int y, int z, T component) {
        for (NetworkType networkType : component.getNetworkTypes()) {
            Network net = getAt(world.field_216, x, y, z, networkType.getIdentifier());

            if (net == null) {
                NyaLib.LOGGER.warn("Removed a block at [x={}, y={}, z={}] which was not in any network of type {}.", x, y, z, networkType.toString());
                continue;
            }

            ArrayList<class_63> neighborBlocks = new ArrayList<>();
            for (Direction direction : Direction.values()) {
                var neighborPos = new class_63(x + direction.getOffsetX(), y + direction.getOffsetY(), z + direction.getOffsetZ());
                if (net.isAt(neighborPos.field_1482, neighborPos.field_1483, neighborPos.field_1484)) {
                    neighborBlocks.add(neighborPos);
                }
            }

            NyaLib.LOGGER.debug("NET SIZE: {} | NEIGHBORS FOUND : {}", net.blocks.size(), neighborBlocks.size());

            switch (neighborBlocks.size()) {
                // There are no neighbors, which should mean that this is the last block in the network and the network can be removed
                case 0 -> {
                    NyaLib.LOGGER.debug("Last block in network, removing network with ID {}", net.getId());
                    net.removeBlock(x, y, z);

                    if (net.blocks.isEmpty()) {
                        removeNetwork(net);
                    } else {
                        NyaLib.LOGGER.warn("Removed a block from network {} with no neighbors but the network still has {} blocks. Network will not be deleted", net.getId(), net.blocks.size());
                    }

                }

                case 1 -> {
                    NyaLib.LOGGER.debug("Only one neighbor, its a stump and can be removed normally");
                    net.removeBlock(x, y, z);
                }

                default -> {
                    net.removeBlock(x, y, z);

                    ArrayList<HashSet<class_63>> potentialNetworks = new ArrayList<>(6);
                    // Walk thru all the sides
                    for (Direction dir : Direction.values()) {
                        class_63 neighbor = new class_63(x + dir.getOffsetX(), y + dir.getOffsetY(), z + dir.getOffsetZ());

                        // If the network reaches this neighbor side, walk thru all the blocks
                        if (net.isAt(neighbor.field_1482, neighbor.field_1483, neighbor.field_1484)) {
                            HashSet<class_63> discovered = net.walk(neighbor);

                            NyaLib.LOGGER.debug("Discovered a potential network of {} blocks", discovered.size());

                            boolean exists = false;

                            // Check if the first block of this potential networks exists in the other potential networks
                            // We dont have to check every block because if theyre connected
                            // somewhere they *should* have access to the same block
                            for (var potentialNet : potentialNetworks) {
                                if (potentialNet.contains(discovered.iterator().next())) {
                                    exists = true;
                                }
                            }

                            // If it doesnt exist, we can safely assume this is an independed new network
                            if (!exists) {
                                potentialNetworks.add(discovered);
                            }
                        }
                    }

                    NyaLib.LOGGER.debug("There will be {} new networks", potentialNetworks.size());

                    switch (potentialNetworks.size()) {
                        // This shouldnt happen
                        case 0 -> {
                            NyaLib.LOGGER.warn("There were 0 potential networks when splitting, this shouldn't happen");
                        }

                        case 1 -> {
                            // This is fine
                        }

                        // 2 or more networks
                        // The first potential network will be kept in the existing one while others will be transferred to a new one
                        // `net` is the first network here
                        default -> {
                            // Iterate over the new potential networks
                            for (int i = 1; i < potentialNetworks.size(); i++) {

                                Network newNetwork = createNetwork(world.field_216, networkType);

                                // Iterate over every block in this new potential network
                                for (class_63 pos : potentialNetworks.get(i)) {
                                    net.removeBlock(pos.field_1482, pos.field_1483, pos.field_1484);
                                    newNetwork.addBlock(
                                            pos.field_1482,
                                            pos.field_1483,
                                            pos.field_1484,
                                            world.getBlockState(pos.field_1482, pos.field_1483, pos.field_1484).getBlock()
                                    );
                                }

                                // Update the new network
                                newNetwork.update();
                            }

                            // Update the existing network
                            net.update();
                        }
                    }
                }
            }

        }
    }

    // Load & Save
    public static void writeNbt(class_18 world, class_8 nbt) {
        class_50 dim = world.field_216;
        Optional<Identifier> dimIdentifierO = DimensionRegistry.INSTANCE.getIdByLegacyId(dim.field_2179);
        Identifier dimIdentifier;

        // Check if a dimension with this ID exists
        if (dimIdentifierO.isEmpty()) {
            NyaLib.LOGGER.error("Dimension {} not found", dim.field_2179);
            return;
        }
        dimIdentifier = dimIdentifierO.get();

        // Get the compound containing all the compounds for each dimension
        if (!nbt.method_1023("dimensions")) {
            nbt.method_1018("dimensions", new class_8());
        }
        class_8 dimensionsNbt = nbt.method_1033("dimensions");

        // Get the compound of the network types in the specified dimension
        if (!dimensionsNbt.method_1023(dimIdentifier.toString())) {
            dimensionsNbt.method_1018(dimIdentifier.toString(), new class_8());
        }
        class_8 dimensionNbt = dimensionsNbt.method_1033(dimIdentifier.toString());

        // Get all the networks in this dimension and iterate over them
        NETWORKS.computeIfAbsent(dim, k -> new HashMap<>());
        for (Map.Entry<Identifier, ArrayList<Network>> networksOfType : getNetworks(dim).entrySet()) {
            Identifier type = networksOfType.getKey();
            var networks = networksOfType.getValue();

            // Get all the networks of the given type
            dimensionNbt.method_1018(type.toString(), new class_8());
            class_8 networksOfTypeNbt = dimensionNbt.method_1033(type.toString());

            // Write Each network
            for (Network network : networks) {
                networksOfTypeNbt.method_1018(network.id + "", network.toNbt());
            }
        }

        // Remove queued up networks on save
        removeQueuedNetworks();
    }

    public static void readNbt(class_18 world, class_8 nbt) {
        class_50 dim = world.field_216;
        Optional<Identifier> dimIdentifierO = DimensionRegistry.INSTANCE.getIdByLegacyId(dim.field_2179);
        Identifier dimIdentifier;

        // Check if a dimension with this ID exists
        if (dimIdentifierO.isEmpty()) {
            NyaLib.LOGGER.error("Dimension {} not found", dim.field_2179);
            return;
        }
        dimIdentifier = dimIdentifierO.get();

        // Get the compound containing all the compounds for each dimension
        class_8 dimensionsNbt = nbt.method_1033("dimensions");

        // Get the compound if the dimension
        class_8 dimensionNbt = dimensionsNbt.method_1033(dimIdentifier.toString());

        // Load the Networks
        for (Object networksOfTypeO : dimensionNbt.method_1024()) {
            if (networksOfTypeO instanceof class_8 networksOfType) {

                // Loading a network for a give type
                for (Object networksO : networksOfType.method_1024()) {
                    if (networksO instanceof class_8 networks) {
                        addNetwork(
                                dim,
                                Network.fromNbt(networks, world, Identifier.of(networksOfType.method_626()))
                        );
                    }
                }
            }
        }

    }

}
