package net.danygames2014.nyalib.item;

import net.minecraft.class_124;
import net.minecraft.class_31;

/**
 * An Item Handler interface to be implemented on {@link net.minecraft.class_57}
 */
@SuppressWarnings("unused")
public interface ItemHandlerEntity {
    /**
     * Check if the entity supports extracting items, if this returns false there
     * should be no point in trying to use any <code>extractItem</code> method.
     *
     * @return <code>true</code> if the entity supports item extraction
     */
    boolean canExtractItem();

    /**
     * Extract any item from any slot
     *
     * @return The extracted ItemStack, <code>null</code> if nothing is extracted
     */
    default class_31 extractItem() {
        for (int i = 0; i < getItemSlots(); i++) {
            if (getItem(i) != null) {
                return extractItem(i, Integer.MAX_VALUE);
            }
        }
        return null;
    }

    /**
     * Extract a specified amount of any item from any slot
     *
     * @param amount The amount of items to extract (can be larger than the maximum stack size)
     * @return The extracted ItemStack, <code>null</code> if nothing is extracted
     */
    default class_31 extractItem(int amount) {
        for (int i = 0; i < getItemSlots(); i++) {
            if (getItem(i) != null) {
                return extractItem(i, amount);
            }
        }
        return null;
    }

    /**
     * Extract an item in the given slot
     *
     * @param slot   The slot to extract from
     * @param amount The amount to extract (can be larger than the maximum stack size)
     * @return The extracted {@link class_31}, <code>null</code> if nothing is extracted
     */
    class_31 extractItem(int slot, int amount);

    /**
     * Extract the given {@link class_124} with any meta from any slot
     *
     * @param item   The Item to extract
     * @param amount The amount to extract (can be larger than the maximum stack size)
     * @return The extracted ItemStack, <code>null</code> if nothing is extracted
     */
    default class_31 extractItem(class_124 item, int amount) {
        return extractItem(item, -1, amount);
    }

    /**
     * Extract the given {@link class_124} with the specified meta from any slot
     *
     * @param item   The Item to extract
     * @param meta   The meta of the item to extract (-1 to match any meta)
     * @param amount The amount to extract (can be larger than the maximum stack size)
     * @return The extracted ItemStack, <code>null</code> if nothing is extracted
     */
    default class_31 extractItem(class_124 item, int meta, int amount) {
        class_31 currentStack = null;
        int remaining = amount;

        for (int i = 0; i < getItemSlots(); i++) {
            if (remaining <= 0) {
                break;
            }

            if (currentStack != null) {
                if (this.getItem(i).method_702(currentStack)) {
                    class_31 extractedStack = extractItem(i, remaining);
                    remaining -= extractedStack.field_751;
                    currentStack.field_751 += extractedStack.field_751;
                }
            } else {
                if (getItem(i).isOf(item) && (meta == -1 || getItem(i).method_722() == meta)) {
                    class_31 extractedStack = extractItem(i, remaining);
                    remaining -= extractedStack.field_751;
                    currentStack = extractedStack;
                }
            }
        }

        return currentStack;
    }

    /**
     * Check if the entity supports inserting items, if this returns false there
     * should be no point in trying to use any <code>insertItem</code> method.
     *
     * @return <code>true</code> if the entity supports item insertion
     */
    boolean canInsertItem();

    /**
     * Insert item into the given slot and return the remainder
     *
     * @param stack The {@link class_31} to insert
     * @param slot  Slot to insert into
     * @return The remainder of the ItemStack (<code>null</code> if it was inserted entirely), this should be a new ItemStack, however it can be the same if it was not modified
     */
    class_31 insertItem(class_31 stack, int slot);

    /**
     * Insert item into any slot and return the remainder
     *
     * @param stack The {@link class_31} to insert
     * @return The remainder of the ItemStack (<code>null</code> if it was inserted entirely), this should be a new ItemStack, however it can be the same if it was not modified
     */
    class_31 insertItem(class_31 stack);

    /**
     * Get the {@link class_31} in the given slot, If there is no {@link class_31}, then return null
     * <p>
     * <bold>DO NOT MODIFY THIS ITEMSTACK</bold>
     *
     * @param slot The slot to get the {@link class_31} from
     * @return The {@link class_31} in the slot, <code>null</code> if the slot is empty
     */
    class_31 getItem(int slot);

    /**
     * Directly sets the given slot to the given ItemStack
     *
     * @param stack The {@link class_31} to set the slot to
     * @param slot  The slot to put the stack into
     * @return Whether the action was succesfull
     */
    boolean setItem(class_31 stack, int slot);

    /**
     * Get the size of the entities's inventory
     *
     * @return The number of slots this entities's inventory has
     */
    int getItemSlots();

    /**
     * Get the entire inventory of the entity
     *
     * @return An array of all the ItemStacks
     */
    class_31[] getInventory();
}
