package net.glasslauncher.mods.alwaysmoreitems.transfer;

import net.glasslauncher.mods.alwaysmoreitems.api.gui.GuiItemStackGroup;
import net.glasslauncher.mods.alwaysmoreitems.api.gui.RecipeLayout;
import net.glasslauncher.mods.alwaysmoreitems.api.recipe.transfer.RecipeTransferError;
import net.glasslauncher.mods.alwaysmoreitems.api.recipe.transfer.RecipeTransferHandler;
import net.glasslauncher.mods.alwaysmoreitems.api.recipe.transfer.RecipeTransferHandlerHelper;
import net.glasslauncher.mods.alwaysmoreitems.api.recipe.transfer.RecipeTransferInfo;
import net.glasslauncher.mods.alwaysmoreitems.gui.widget.ingredients.IGuiIngredient;
import net.glasslauncher.mods.alwaysmoreitems.network.c2s.RecipeTransferPacket;
import net.glasslauncher.mods.alwaysmoreitems.util.AlwaysMoreItems;
import net.glasslauncher.mods.alwaysmoreitems.util.StackHelper;
import net.minecraft.class_133;
import net.minecraft.class_300;
import net.minecraft.class_31;
import net.minecraft.class_54;
import net.minecraft.class_71;
import net.modificationstation.stationapi.api.network.packet.PacketHelper;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.*;

public class BasicRecipeTransferHandler implements RecipeTransferHandler {
	@Nonnull
	private final RecipeTransferInfo transferHelper;

	public BasicRecipeTransferHandler(@Nonnull RecipeTransferInfo transferHelper) {
		this.transferHelper = transferHelper;
	}

	@Override
	public Class<? extends class_71> getContainerClass() {
		return transferHelper.getContainerClass();
	}

	@Override
	public String getRecipeCategoryUid() {
		return transferHelper.getRecipeCategoryUid();
	}

	@Nullable
	@Override
	public RecipeTransferError transferRecipe(@Nonnull class_71 container, @Nonnull RecipeLayout recipeLayout, @Nonnull class_54 player, boolean maxTransfer, boolean doTransfer) {
		RecipeTransferHandlerHelper handlerHelper = AlwaysMoreItems.getHelpers().recipeTransferHandlerHelper();
		StackHelper stackHelper = AlwaysMoreItems.getStackHelper();

		if (!AlwaysMoreItems.isAMIOnServer()) {
			return handlerHelper.createInternalError();
		}

		Map<Integer, class_133> inventorySlots = new HashMap<>();
		for (class_133 slot : transferHelper.getInventorySlots(container)) {
			inventorySlots.put(slot.field_499, slot);
		}

		Map<Integer, class_133> craftingSlots = new HashMap<>();
		for (class_133 slot : transferHelper.getRecipeSlots(container)) {
			craftingSlots.put(slot.field_499, slot);
		}

		int inputCount = 0;
		GuiItemStackGroup itemStackGroup = recipeLayout.getItemStacks();
		for (IGuiIngredient<class_31> ingredient : itemStackGroup.getGuiIngredients().values()) {
			if (ingredient.isInput() && !ingredient.getAll().isEmpty()) {
				inputCount++;
			}
		}

		if (inputCount > craftingSlots.size()) {
			AlwaysMoreItems.LOGGER.error("Recipe Transfer helper {} does not work for container {}", transferHelper.getClass(), container.getClass());
			return handlerHelper.createInternalError();
		}

		List<class_31> availableItemStacks = new ArrayList<>();
		int filledCraftSlotCount = 0;
		int emptySlotCount = 0;

		for (class_133 slot : craftingSlots.values()) {
			if (slot.method_476()) {
				if (slot.method_473(0) == null) { // TODO: Do this better.
					AlwaysMoreItems.LOGGER.error("Recipe Transfer helper {} does not work for container {}. Player can't move item out of Crafting Slot number {}", transferHelper.getClass(), container.getClass(), slot.field_499);
					return handlerHelper.createInternalError();
				}
				filledCraftSlotCount++;
				availableItemStacks.add(slot.method_472().method_724());
			}
		}

		for (class_133 slot : inventorySlots.values()) {
			if (slot.method_476()) {
				availableItemStacks.add(slot.method_472().method_724());
			} else {
				emptySlotCount++;
			}
		}

		// check if we have enough inventory space to shuffle items around to their final locations
		if (filledCraftSlotCount - inputCount > emptySlotCount) {
			String message = class_300.method_992().method_993("alwaysmoreitems.tooltip.error.recipe.transfer.inventory.full");
			return handlerHelper.createUserErrorWithTooltip(message);
		}

		StackHelper.MatchingItemsResult matchingItemsResult = stackHelper.getMatchingItems(availableItemStacks, itemStackGroup.getGuiIngredients());

		if (!matchingItemsResult.missingItems.isEmpty()) {
			String message = class_300.method_992().method_993("alwaysmoreitems.tooltip.error.recipe.transfer.missing");
			return handlerHelper.createUserErrorForSlots(message, matchingItemsResult.missingItems);
		}

		List<Integer> craftingSlotIndexes = new ArrayList<>(craftingSlots.keySet());
		Collections.sort(craftingSlotIndexes);

		List<Integer> inventorySlotIndexes = new ArrayList<>(inventorySlots.keySet());
		Collections.sort(inventorySlotIndexes);

		// check that the slots exist and can be altered
		for (Map.Entry<Integer, class_31> entry : matchingItemsResult.matchingItems.entrySet()) {
			int craftNumber = entry.getKey();
			int slotNumber = craftingSlotIndexes.get(craftNumber);
			if (slotNumber >= container.field_2734.size()) {
				AlwaysMoreItems.LOGGER.error("Recipes Transfer Helper {} references slot {} outside of the inventory's size {}", transferHelper.getClass(), slotNumber, container.field_2734.size());
				return handlerHelper.createInternalError();
			}
			class_133 slot = container.method_2084(slotNumber);
			class_31 stack = entry.getValue();
			if (slot == null) {
				AlwaysMoreItems.LOGGER.error("The slot number {} does not exist in the container.", slotNumber);
				return handlerHelper.createInternalError();
			}
			if (!slot.method_477(stack)) {
				AlwaysMoreItems.LOGGER.error("The ItemStack {} is not valid for the slot number {}", stack, slotNumber);
				return handlerHelper.createInternalError();
			}
		}

		if (doTransfer) {
			RecipeTransferPacket packet = new RecipeTransferPacket(matchingItemsResult.matchingItems, craftingSlotIndexes, inventorySlotIndexes, maxTransfer);
			PacketHelper.send(packet);
		}

		return null;
	}

	public static void setItems(@Nonnull class_54 player, @Nonnull Map<Integer, class_31> slotMap, @Nonnull List<Integer> craftingSlots, @Nonnull List<Integer> inventorySlots, boolean maxTransfer) {
		class_71 container = player.field_521;
		StackHelper stackHelper = AlwaysMoreItems.getStackHelper();

		// remove required recipe items
		int removedSets = removeSetsFromInventory(container, slotMap.values(), craftingSlots, inventorySlots, maxTransfer);
		if (removedSets == 0) {
			return;
		}

		// clear the crafting grid
		List<class_31> clearedCraftingItems = new ArrayList<>();
		for (Integer craftingSlotNumber : craftingSlots) {
			class_133 craftingSlot = container.method_2084(craftingSlotNumber);
			if (craftingSlot != null && craftingSlot.method_476()) {
				class_31 craftingItem = craftingSlot.method_473(Integer.MAX_VALUE);
				clearedCraftingItems.add(craftingItem);
			}
		}

		// put items into the crafting grid
		for (Map.Entry<Integer, class_31> entry : slotMap.entrySet()) {
			class_31 stack = entry.getValue();
			if (stack.method_715()) {
				int maxSets = stack.method_709() / stack.field_751;
				stack.field_751 *= Math.min(maxSets, removedSets);
			}
			Integer craftNumber = entry.getKey();
			Integer slotNumber = craftingSlots.get(craftNumber);
			class_133 slot = container.method_2084(slotNumber);
			slot.method_479(stack);
		}

		// put cleared items back into the player's inventory
		for (class_31 oldCraftingItem : clearedCraftingItems) {
			stackHelper.addStack(container, inventorySlots, oldCraftingItem, true);
		}

		container.method_2075();
	}

	private static int removeSetsFromInventory(@Nonnull class_71 container, @Nonnull Collection<class_31> required, @Nonnull List<Integer> craftingSlots, @Nonnull List<Integer> inventorySlots, boolean maxTransfer) {
		if (maxTransfer) {
            List<class_31> requiredCopy = new ArrayList<>(required);

			int removedSets = 0;
			while (!requiredCopy.isEmpty() && removeSetsFromInventory(container, requiredCopy, craftingSlots, inventorySlots)) {
				removedSets++;
				Iterator<class_31> iterator = requiredCopy.iterator();
				while (iterator.hasNext()) {
					class_31 stack = iterator.next();
					if (!stack.method_715() || (stack.field_751 * (removedSets + 1) > stack.method_709())) {
						iterator.remove();
					}
				}
			}
			return removedSets;
		} else {
			boolean success = removeSetsFromInventory(container, required, craftingSlots, inventorySlots);
			return success ? 1 : 0;
		}
	}

	private static boolean removeSetsFromInventory(@Nonnull class_71 container, @Nonnull Iterable<class_31> required, @Nonnull List<Integer> craftingSlots, @Nonnull List<Integer> inventorySlots) {
		final Map<class_133, class_31> originalSlotContents = new HashMap<>();

		for (class_31 matchingStack : required) {
			final class_31 requiredStack = matchingStack.method_724();
			while (requiredStack.field_751 > 0) {
				final class_133 slot = getSlotWithStack(container, requiredStack, craftingSlots, inventorySlots);
				if (slot == null) {
					// abort! put removed items back where the came from
					for (Map.Entry<class_133, class_31> slotEntry : originalSlotContents.entrySet()) {
						class_31 stack = slotEntry.getValue();
						slotEntry.getKey().method_479(stack);
					}
					return false;
				}

				if (!originalSlotContents.containsKey(slot)) {
					originalSlotContents.put(slot, slot.method_472().method_724());
				}

				class_31 removed = slot.method_473(requiredStack.field_751);
				requiredStack.field_751 -= removed.field_751;
			}
		}

		return true;
	}

	private static class_133 getSlotWithStack(@Nonnull class_71 container, @Nonnull class_31 stack, @Nonnull List<Integer> craftingSlots, @Nonnull List<Integer> inventorySlots) {
		StackHelper stackHelper = AlwaysMoreItems.getStackHelper();

		class_133 slot = stackHelper.getSlotWithStack(container, craftingSlots, stack);
		if (slot == null) {
			slot = stackHelper.getSlotWithStack(container, inventorySlots, stack);
		}

		return slot;
	}
}
