package net.glasslauncher.mods.alwaysmoreitems.registry;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableListMultimap;
import net.fabricmc.loader.api.FabricLoader;
import net.fabricmc.loader.api.ModContainer;
import net.glasslauncher.mods.alwaysmoreitems.api.ItemRegistry;
import net.glasslauncher.mods.alwaysmoreitems.util.AlwaysMoreItems;
import net.minecraft.class_124;
import net.minecraft.class_31;
import net.modificationstation.stationapi.api.recipe.FuelRegistry;
import net.modificationstation.stationapi.api.util.Identifier;

import javax.annotation.*;
import java.util.*;

public class AMIItemRegistry implements ItemRegistry {

    @Nonnull
    private final Set<String> itemNameSet = new HashSet<>();
    @Nonnull
    private final ImmutableList<class_31> itemList;
    @Nonnull
    private final ImmutableListMultimap<String, class_31> itemsByModId;
    @Nonnull
    private final ImmutableList<class_31> fuels;

    public AMIItemRegistry() {
        List<class_31> itemListMutable = new ArrayList<>();
        List<class_31> fuelsMutable = new ArrayList<>();

        class_124[] items = net.modificationstation.stationapi.api.registry.ItemRegistry.INSTANCE.stream().toArray(class_124[]::new);
        for (class_124 item : items) {
            addItemAndSubItems(item, itemListMutable, fuelsMutable);
        }

        this.itemList = ImmutableList.copyOf(itemListMutable);
        this.fuels = ImmutableList.copyOf(fuelsMutable);

        ImmutableListMultimap.Builder<String, class_31> itemsByModIdBuilder = ImmutableListMultimap.builder();
        for (class_31 itemStack : itemListMutable) {
            class_124 item = itemStack.method_694();
            if (item != null) {
                Identifier itemResourceLocation = net.modificationstation.stationapi.api.registry.ItemRegistry.INSTANCE.getId(itemStack.method_694());
                if (itemResourceLocation == null) {
                    AlwaysMoreItems.LOGGER.warn("Item has no associated mod id", new NullPointerException());
                    continue;
                }
                String modId = itemResourceLocation.namespace.toString().toLowerCase(Locale.ENGLISH);
                itemsByModIdBuilder.put(modId, itemStack);
            }
        }
        this.itemsByModId = itemsByModIdBuilder.build();
    }

    @Override
    @Nonnull
    public ImmutableList<class_31> getItemList() {
        return itemList;
    }

    @Override
    @Nonnull
    public ImmutableList<class_31> getFuels() {
        return fuels;
    }

    @Nonnull
    @Override
    public String getModNameForItem(@Nullable class_124 item) {
        Identifier identifier = net.modificationstation.stationapi.api.registry.ItemRegistry.INSTANCE.getId(item);
        if (identifier == null) {
            AlwaysMoreItems.LOGGER.error("Item has no identifier?", new NullPointerException());
            return "";
        }
        Optional<ModContainer> modContainer = FabricLoader.getInstance().getModContainer(identifier.namespace.toString());
        if (modContainer.isEmpty()) {
            AlwaysMoreItems.LOGGER.error("Mod namespace has no container", new NullPointerException());
            return "";
        }
        return modContainer.get().getMetadata().getName();
    }

    @Nonnull
    @Override
    public ImmutableList<class_31> getItemListForModId(@Nullable String modId) {
        if (modId == null) {
            AlwaysMoreItems.LOGGER.error("Null modId", new NullPointerException());
            return ImmutableList.of();
        }
        String lowerCaseModId = modId.toLowerCase(Locale.ENGLISH);
        return itemsByModId.get(lowerCaseModId);
    }

    private void addItemAndSubItems(@Nullable class_124 item, @Nonnull List<class_31> itemList, @Nonnull List<class_31> fuels) {
        if (item == null) {
            return;
        }

        List<class_31> items = AlwaysMoreItems.getStackHelper().getSubtypes(item, 1);
        addItemStacks(items, itemList, fuels);
    }

    private void addItemStacks(@Nonnull Iterable<class_31> stacks, @Nonnull List<class_31> itemList, @Nonnull List<class_31> fuels) {
        for (class_31 stack : stacks) {
            if (stack != null) {
                addItemStack(stack, itemList, fuels);
            }
        }
    }

    private void addItemStack(@Nonnull class_31 stack, @Nonnull List<class_31> itemList, @Nonnull List<class_31> fuels) {
        try {
            String itemKey = AlwaysMoreItems.getStackHelper().getUniqueIdentifierForStack(stack);

            if (itemNameSet.contains(itemKey)) {
                return;
            }
            itemNameSet.add(itemKey);
            itemList.add(stack);

            if (FuelRegistry.getFuelTime(stack) > 0) {
                fuels.add(stack);
            }
        } catch (RuntimeException e) {
            AlwaysMoreItems.LOGGER.error("Couldn't create unique name for itemStack {}.", stack.getClass(), e);
        }
    }
}