package net.glasslauncher.mods.alwaysmoreitems.util;

import com.mojang.datafixers.util.Either;
import net.fabricmc.api.EnvType;
import net.fabricmc.loader.api.FabricLoader;
import net.glasslauncher.mods.alwaysmoreitems.api.AMINbt;
import net.glasslauncher.mods.alwaysmoreitems.api.SubItemHelper;
import net.glasslauncher.mods.alwaysmoreitems.config.AMIConfig;
import net.glasslauncher.mods.alwaysmoreitems.gui.widget.ingredients.IGuiIngredient;
import net.minecraft.class_124;
import net.minecraft.class_133;
import net.minecraft.class_31;
import net.minecraft.class_533;
import net.minecraft.class_629;
import net.minecraft.class_71;
import net.minecraft.class_8;
import net.modificationstation.stationapi.api.block.HasMetaNamedBlockItem;
import net.modificationstation.stationapi.api.block.MetaNamedBlockItemProvider;
import net.modificationstation.stationapi.api.registry.ItemRegistry;
import net.modificationstation.stationapi.api.registry.RegistryEntry;
import net.modificationstation.stationapi.api.tag.TagKey;
import net.modificationstation.stationapi.api.util.Identifier;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.SortedSet;
import java.util.TreeSet;

public class StackHelper implements net.glasslauncher.mods.alwaysmoreitems.api.recipe.StackHelper {
    /**
     * Returns a list of items in slots that complete the recipe defined by requiredStacksList.
     * Returns a result that contains missingItems if there are not enough items in availableItemStacks.
     */
    @Nonnull
    public MatchingItemsResult getMatchingItems(@Nonnull List<class_31> availableItemStacks, @Nonnull Map<Integer, ? extends IGuiIngredient<class_31>> ingredientsMap) {
        MatchingItemsResult matchingItemResult = new MatchingItemsResult();

        int recipeSlotNumber = -1;
        SortedSet<Integer> keys = new TreeSet<>(ingredientsMap.keySet());
        for (Integer key : keys) {
            IGuiIngredient<class_31> ingredient = ingredientsMap.get(key);
            if (!ingredient.isInput()) {
                continue;
            }
            recipeSlotNumber++;

            List<class_31> requiredStacks = ingredient.getAll();
            if (requiredStacks.isEmpty()) {
                continue;
            }

            class_31 matching = containsStack(availableItemStacks, requiredStacks);
            if (matching == null) {
                matchingItemResult.missingItems.add(key);
            } else {
                class_31 matchingSplit = matching.method_695(1);
                if (matching.field_751 == 0) {
                    availableItemStacks.remove(matching);
                }
                matchingItemResult.matchingItems.put(recipeSlotNumber, matchingSplit);
            }
        }

        return matchingItemResult;
    }

    @Nullable
    public class_133 getSlotWithStack(@Nonnull class_71 container, @Nonnull Iterable<Integer> slotNumbers, @Nonnull class_31 stack) {
        StackHelper stackHelper = AlwaysMoreItems.getStackHelper();

        for (Integer slotNumber : slotNumbers) {
            class_133 slot = container.method_2084(slotNumber);
            if (slot != null) {
                class_31 slotStack = slot.method_472();
                if (stackHelper.isIdentical(stack, slotStack)) {
                    return slot;
                }
            }
        }
        return null;
    }

    @Nonnull
    public List<class_31> removeDuplicateItemStacks(@Nonnull Iterable<class_31> stacks) {
        List<class_31> newStacks = new ArrayList<>();
        for (class_31 stack : stacks) {
            if (stack != null && containsStack(newStacks, stack) == null) {
                newStacks.add(stack);
            }
        }
        return newStacks;
    }

    /* Returns an ItemStack from "stacks" if it isIdentical to an ItemStack from "contains" */
    @Nullable
    public class_31 containsStack(@Nullable Iterable<class_31> stacks, @Nullable Iterable<class_31> contains) {
        if (stacks == null || contains == null) {
            return null;
        }

        for (class_31 containStack : contains) {
            class_31 matchingStack = containsStack(stacks, containStack);
            if (matchingStack != null) {
                return matchingStack;
            }
        }

        return null;
    }

    /* Returns an ItemStack from "stacks" if it isIdentical to "contains" */
    @Nullable
    public class_31 containsStack(@Nullable Iterable<class_31> stacks, @Nullable class_31 contains) {
        if (stacks == null || contains == null) {
            return null;
        }

        for (class_31 stack : stacks) {
            if (isIdentical(contains, stack)) {
                return stack;
            }
        }
        return null;
    }

    public boolean isIdentical(@Nullable class_31 lhs, @Nullable class_31 rhs) {
        if (lhs == rhs) {
            return true;
        }

        if (lhs == null || rhs == null) {
            return false;
        }

        if (lhs.method_694() != rhs.method_694()) {
            return false;
        }

        if (lhs.method_722() != -1/*OreDictionary.WILDCARD_VALUE*/) {
            if (lhs.method_722() != rhs.method_722()) {
                return false;
            }
        }

        return lhs.method_702(rhs);
    }

    @Override
    @Nonnull
    public List<class_31> getSubtypes(@Nullable class_31 itemStack) {
        if (itemStack == null) {
            AlwaysMoreItems.LOGGER.error("Null itemStack", new NullPointerException());
            return Collections.emptyList();
        }

        class_124 item = itemStack.method_694();
        if (item == null) {
            AlwaysMoreItems.LOGGER.error("Null item in itemStack", new NullPointerException());
            return Collections.emptyList();
        }

        if (itemStack.method_722() != -1/*OreDictionary.WILDCARD_VALUE*/) {
            return new ArrayList<>() {
                {
                    add(itemStack);
                }
            };
        }

        return getSubtypes(item, itemStack.field_751);
    }

    @Nonnull
    public List<class_31> getSubtypes(@Nonnull class_124 item, int stackSize) {
//        List<ItemStack> itemStacks = new ArrayList<>();

//        for (CreativeTabs itemTab : item.getCreativeTabs()) {
//            List<ItemStack> subItems = new ArrayList<>();
//            item.getSubItems(item, itemTab, subItems);
//            for (ItemStack subItem : subItems) {
//                if (subItem.stackSize != stackSize) {
//                    ItemStack subItemCopy = subItem.copy();
//                    subItemCopy.stackSize = stackSize;
//                    itemStacks.add(subItemCopy);
//                } else {
//                    itemStacks.add(subItem);
//                }
//            }
//        }

        // Try to get the mod defined sub items
        List<class_31> subItems = SubItemHelper.getSubItems(item);

        // If mod has defined sub items, use those
        if (subItems != null && !subItems.isEmpty()) {
            subItems = subItems.stream().peek(itemStack -> itemStack.field_751 = stackSize).toList();
        }

        // If mod hasnt defined sub items, look for them ourselves
        else {
            // Create a list for sub items
            subItems = new ArrayList<>();
            List<String> keyCache = new ArrayList<>();

            // If on server, we dont care, generate first item and return
            if (FabricLoader.getInstance().getEnvironmentType().equals(EnvType.SERVER)) {
                subItems.add(new class_31(item, stackSize, 0));
                return subItems;
            }

            // If item is a BlockItem and that BlockItem has MetaNamedBlockItemProvider, use that and return
            if (item instanceof class_533 blockItem) {
                // StationAPI interface
                if (blockItem.getBlock() instanceof MetaNamedBlockItemProvider metaBlockItemProvider) {
                    for (int i = 0; i < metaBlockItemProvider.getValidMetas().length; i++) {
                        subItems.add(new class_31(item, stackSize, metaBlockItemProvider.getValidMetas()[i]));
                    }
                    return subItems;
                }

                // StationAPI annotation
                if (blockItem.getBlock() instanceof HasMetaNamedBlockItem blockItemWithMeta) {
                    for (int i = 0; i < blockItemWithMeta.validMetas().length; i++) {
                        subItems.add(new class_31(item, stackSize, blockItemWithMeta.validMetas()[i]));
                    }
                    return subItems;
                }
            }

            // As a last resort try to scan all the 16 possible meta values
            for (int meta = 0; meta < 16; meta++) {
                try { // Shitcoders go brrr
                    class_31 itemStack = new class_31(item, stackSize, meta);
                    String translationKey = itemStack.method_726();

                    // If this Translation Key has already been observed, ignore it
                    if (keyCache.contains(translationKey + "@" + itemStack.method_725())) {
                        continue;
                    }

                    // Check if the name and ends with the meta (like the aether dart shooter)
                    if (translationKey.endsWith(String.valueOf(meta))) {
                        // If meta is present, query for the translation with that meta
                        String translatedNameWithMeta = class_629.method_2049(translationKey + ".name");

                        // If translatedName is not translated and is the raw translation key, 
                        // then removing the last 5 characters will remove ".name" allowing the comparison with the translationKey 
                        if (translationKey.contains(translatedNameWithMeta.substring(0, translatedNameWithMeta.length() - 5))) {
                            // This meta is not translated, avoid
                            AlwaysMoreItems.LOGGER.debug("Untranslated meta value {} hidden, translation key is {}", meta, translationKey);
                            keyCache.add(translationKey + "@" + itemStack.method_725());
                            continue;
                        }
                    }

                    // Check if the item does not have a translation key
                    if (itemStack.method_694().method_469().equals(translationKey + ".name")) {
                        AlwaysMoreItems.LOGGER.debug("Item {} is not translated", translationKey);

                        // Check if ignoring untranslated names is enabled
                        if (AMIConfig.ignoreUntranslatedNames()) {
                            // Add only the item with meta 0 and return
                            subItems.add(new class_31(item, stackSize, meta));
                            return subItems;
                        }
                    }

                    keyCache.add(translationKey + "@" + itemStack.method_725());
                    subItems.add(itemStack);
                } catch (Exception e) {
                    //noinspection OptionalGetWithoutIsPresent if we throw an exception on this get, then someone fucked up big time.
                    AlwaysMoreItems.LOGGER.error("An item being autoregistered threw an exception, yell at the creator of " + ItemRegistry.INSTANCE.getId(item.field_461).get(), e);
                }
            }
        }

        return subItems;
    }

    @Override
    @Nonnull
    public List<class_31> getAllSubtypes(@Nullable Iterable stacks) {
        if (stacks == null) {
            AlwaysMoreItems.LOGGER.error("Null stacks", new NullPointerException());
            return Collections.emptyList();
        }

        List<class_31> allSubtypes = new ArrayList<>();
        getAllSubtypes(allSubtypes, stacks);
        return allSubtypes;
    }

    private void getAllSubtypes(@Nonnull List<class_31> subtypesList, @Nonnull Iterable stacks) {
        for (Object obj : stacks) {
            if (obj instanceof class_31 stack) {
                List<class_31> subtypes = getSubtypes(stack);
                subtypesList.addAll(subtypes);
            } else if (obj instanceof Iterable iterable) {
                getAllSubtypes(subtypesList, iterable);
            } else if (obj != null) {
                AlwaysMoreItems.LOGGER.error("Unknown object found: {}", obj);
            }
        }
    }

    @Override
    @Nonnull
    public List<class_31> toItemStackList(@Nullable Object stacks) {
        if (stacks == null) {
            return Collections.emptyList();
        }

        List<class_31> itemStacksList = new ArrayList<>();
        toItemStackList(itemStacksList, stacks);
        return removeDuplicateItemStacks(itemStacksList);
    }

    private void toItemStackList(@Nonnull List<class_31> itemStackList, @Nullable Object input) {
        if (input instanceof class_31 stack) {
            itemStackList.add(stack);
        } else if (input instanceof String) {
//            List<ItemStack> stacks = ItemRegistry.INSTANCE.getEntryList(TagKey.of(ItemRegistry.KEY, Identifier.of(string)));
//            itemStackList.addAll(stacks);
        } else if (input instanceof Iterable iterable) {
            for (Object obj : iterable) {
                toItemStackList(itemStackList, obj);
            }
        } else if (input instanceof Either<?, ?> either) {
            if (either.left().isPresent()) { // LEFT = TagKey
                //System.out.println("LEFT " + either.left().get().getClass().getSimpleName());
                if (either.left().get() instanceof TagKey<?> iTagKey) {
                    Optional<TagKey<class_124>> tagKey = iTagKey.tryCast(ItemRegistry.INSTANCE.getKey());
                    if (tagKey.isPresent()) {
                        for (RegistryEntry<class_124> entry : ItemRegistry.INSTANCE.getOrCreateEntryList(tagKey.get())) {
                            itemStackList.add(new class_31(entry.value(), 1));
                        }
                    }
                }
            } else if (either.right().isPresent()) { // Right = ItemStack
                //System.out.println("RIGHT " + either.right().get().getClass().getSimpleName());
                if (either.right().get() instanceof class_31 stack) {
                    itemStackList.add(stack);
                }
            }

        } else if (input != null) {
            AlwaysMoreItems.LOGGER.error("Unknown object found: {}", input);
        }
    }

    @Nonnull
    public String getUniqueIdentifierForStack(@Nonnull class_31 stack) {
        return getUniqueIdentifierForStack(stack, false);
    }

    @Nonnull
    public String getUniqueIdentifierForStack(@Nonnull class_31 stack, boolean wildcard) {
        class_124 item = stack.method_694();
        if (item == null) {
            throw new ItemUidException("Found an itemStack with a null item. This is an error from another mod.");
        }

        Identifier itemName = ItemRegistry.INSTANCE.getId(item);
        if (itemName == null) {
            throw new ItemUidException("No name for item in GameData itemRegistry: " + item.getClass());
        }

        String itemNameString = itemName.toString();
        int metadata = stack.method_722();
        if (wildcard || metadata == -1 /*OreDictionary.WILDCARD_VALUE*/) {
            return itemNameString;
        }

        StringBuilder itemKey = new StringBuilder(itemNameString);
        if (stack.method_719()) {
            itemKey.append(':').append(metadata);
            if (!((AMINbt) stack.getStationNbt()).always_More_Items$hasNoTags()) {
                class_8 nbtTagCompound = AlwaysMoreItems.getHelpers().getNbtIgnoreList().getNbt(stack);
                if (nbtTagCompound != null && !((AMINbt) nbtTagCompound).always_More_Items$hasNoTags()) {
                    itemKey.append(':').append(nbtTagCompound);
                }
            }
        }

        return itemKey.toString();
    }

    @Nonnull
    public List<String> getUniqueIdentifiersWithWildcard(@Nonnull class_31 itemStack) {
        String uid = getUniqueIdentifierForStack(itemStack, false);
        String uidWild = getUniqueIdentifierForStack(itemStack, true);

        if (uid.equals(uidWild)) {
            return Collections.singletonList(uid);
        } else {
            return Arrays.asList(uid, uidWild);
        }
    }

    public int addStack(@Nonnull class_71 container, @Nonnull Collection<Integer> slotIndexes, @Nonnull class_31 stack, boolean doAdd) {
        int added = 0;
        // Add to existing stacks first
        for (Integer slotIndex : slotIndexes) {
            class_133 slot = container.method_2084(slotIndex);
            if (slot == null) {
                continue;
            }

            class_31 inventoryStack = slot.method_472();
            if (inventoryStack == null || inventoryStack.method_694() == null) {
                continue;
            }

            // Already occupied by different item, skip this slot.
            if (!inventoryStack.method_715() || !inventoryStack.method_702(stack) || !class_31.method_703(inventoryStack, stack)) {
                continue;
            }

            int remain = stack.field_751 - added;
            int space = inventoryStack.method_709() - inventoryStack.field_751;
            if (space <= 0) {
                continue;
            }

            // Enough space
            if (space >= remain) {
                if (doAdd) {
                    inventoryStack.field_751 += remain;
                }
                return stack.field_751;
            }

            // Not enough space
            if (doAdd) {
                inventoryStack.field_751 = inventoryStack.method_709();
            }

            added += space;
        }

        if (added >= stack.field_751) {
            return added;
        }

        for (Integer slotIndex : slotIndexes) {
            class_133 slot = container.method_2084(slotIndex);
            if (slot == null) {
                continue;
            }

            class_31 inventoryStack = slot.method_472();
            if (inventoryStack != null) {
                continue;
            }

            if (doAdd) {
                class_31 stackToAdd = stack.method_724();
                stackToAdd.field_751 = stack.field_751 - added;
                slot.method_479(stackToAdd);
            }
            return stack.field_751;
        }

        return added;
    }

    public static class MatchingItemsResult {
        @Nonnull
        public final Map<Integer, class_31> matchingItems = new HashMap<>();
        @Nonnull
        public final List<Integer> missingItems = new ArrayList<>();
    }
}
