package net.zekromaster.minecraft.ironchests.upgrades

import com.google.common.collect.HashBasedTable
import com.google.common.collect.Table
import net.mine_diver.unsafeevents.listener.EventListener
import net.minecraft.block.Block
import net.minecraft.block.entity.ChestBlockEntity
import net.minecraft.entity.player.PlayerEntity
import net.minecraft.item.Item
import net.minecraft.item.ItemStack
import net.minecraft.world.World
import net.modificationstation.stationapi.api.client.item.CustomTooltipProvider
import net.modificationstation.stationapi.api.event.recipe.RecipeRegisterEvent
import net.modificationstation.stationapi.api.event.registry.ItemRegistryEvent
import net.modificationstation.stationapi.api.recipe.CraftingRegistry
import net.modificationstation.stationapi.api.util.Identifier
import net.modificationstation.stationapi.api.util.Namespace
import net.modificationstation.stationapi.api.util.math.Direction
import net.zekromaster.minecraft.ironchests.IronChestsBlockStates.FACING
import net.zekromaster.minecraft.ironchests.IronChestsBlockStates.HAS_OBSIDIAN_UPGRADE
import net.zekromaster.minecraft.terminal.capabilities.BlockCapability
import net.zekromaster.minecraft.terminal.capabilities.CapabilityEvents
import net.zekromaster.minecraft.ironchests.*


interface Tiered {

    val currentTier: IronChestMaterial

    fun switchTier(destination: IronChestMaterial, player: PlayerEntity?): Boolean

    companion object {
        @JvmStatic @get:JvmName("capability")
        val CAPABILITY: BlockCapability<Tiered, Void> = BlockCapability.createVoid(
            Identifier.of("ironchests:tiered"),
            Tiered::class.java
        )
    }
}

private class VanillaChestTieredImpl(val chest: ChestBlockEntity) : Tiered {
    override val currentTier: IronChestMaterial = IronChestMaterial.WOOD

    override fun switchTier(destination: IronChestMaterial, player: PlayerEntity?): Boolean {
        val world = chest.world
        val x = chest.x
        val y = chest.y
        val z = chest.z

        val oldInventory = chest.inventory().copyOf(destination.size)

        world.setBlockStateWithNotify(
            x, y, z,
            destination.block.defaultState.with(FACING, player?.placementFacing() ?: Direction.NORTH)
        )
        (world.getBlockEntity(x, y, z) as IronChestBlockEntity).inventory(oldInventory)
        world.setBlockDirty(x, y, z)
        return true
    }
}

private class IronChestTieredImpl(val chest: IronChestBlockEntity) : Tiered {
    override val currentTier: IronChestMaterial
        get() = chest.material

    override fun switchTier(destination: IronChestMaterial, player: PlayerEntity?): Boolean {
        val world = chest.world
        val x = chest.x
        val y = chest.y
        val z = chest.z

        chest.material = destination

        val originalBlockState = world.getBlockState(x, y, z)

        world.setBlockState(
            x, y, z,
            destination.block.defaultState
                .with(FACING, originalBlockState.get(FACING) ?: Direction.NORTH)
                .with(HAS_OBSIDIAN_UPGRADE, originalBlockState.get(HAS_OBSIDIAN_UPGRADE) ?: false)
        )
        world.setBlockEntity(x, y, z, chest)
        world.setBlockDirty(x, y, z)
        chest.markDirty()
        return true
    }
}

private class TierUpgrade(identifier: Identifier, val starting: IronChestMaterial, val destination: IronChestMaterial):
    ChestUpgrade(identifier), CustomTooltipProvider
{

    override fun innerUpgrade(world: World, x: Int, y: Int, z: Int, player: PlayerEntity, blockEntity: ChestBlockEntity): Boolean {
        val tiered = Tiered.CAPABILITY.get(world, x, y, z, null)
        if (tiered == null || tiered.currentTier != starting) {
            return false
        }

        return tiered.switchTier(destination, player)
    }

    override fun getTooltip(stack: ItemStack, originalTooltip: String) = arrayOf(
        originalTooltip,
        "${starting.displayName} to ${destination.displayName}"
    )

}

object TierUpgradesEntrypoint {

    val upgrades: Table<IronChestMaterial, IronChestMaterial, Item> = HashBasedTable.create()

    @EventListener
    fun registerCapabilities(event: CapabilityEvents.RegisterBlockEntityCapabilitiesEvent) {
        event.register(
            Tiered.CAPABILITY,
            { c, _ -> VanillaChestTieredImpl(c as ChestBlockEntity) },
            "Chest"
        )
        event.register(
            Tiered.CAPABILITY,
            { c, _ -> IronChestTieredImpl(c as IronChestBlockEntity) },
            "IronChest"
        )
    }

    private fun Identifier.toShortString(): String =
        if (namespace === Namespace.MINECRAFT || namespace === NAMESPACE) {
            path
        } else {
            "${namespace}_${path}"
        }.replace("/", "_")

    @EventListener
    fun registerItems(event: ItemRegistryEvent) {
        for (transition in TRANSITIONS.cellSet().filter { it.value.hasUpgrade }) {
            NAMESPACE.id("upgrades/${transition.rowKey.id.toShortString()}_to_${transition.columnKey.id.toShortString()}").apply {
                val upgrade = TierUpgrade(this, transition.rowKey, transition.columnKey)
                upgrade.setTranslationKey(this)
                upgrades.put(transition.rowKey, transition.columnKey, upgrade)
            }
        }
    }

    @EventListener
    fun registerRecipes(event: RecipeRegisterEvent) {
        if (RecipeRegisterEvent.Vanilla.fromType(event.recipeId) != RecipeRegisterEvent.Vanilla.CRAFTING_SHAPED) {
            return
        }

        for (transition in upgrades.cellSet()) {
            val data = TRANSITIONS.get(transition.rowKey, transition.columnKey) ?: continue

            if (data.isHalfStep) {
                CraftingRegistry.addShapedRecipe(
                    ItemStack(transition.value),
                    "igi", "gcg", "igi",
                    'i', transition.columnKey.craftingMaterial,
                    'c', transition.rowKey.craftingMaterial,
                    'g', Block.GLASS
                )
                CraftingRegistry.addShapedRecipe(
                    ItemStack(transition.value),
                    "gig", "ici", "gig",
                    'i', transition.columnKey.craftingMaterial,
                    'c', transition.rowKey.craftingMaterial,
                    'g', Block.GLASS
                )
            } else {
                CraftingRegistry.addShapedRecipe(
                    ItemStack(transition.value),
                    "iii", "ici", "iii",
                    'i', transition.columnKey.craftingMaterial,
                    'c', transition.rowKey.craftingMaterial
                )
            }
        }
    }
}
