package net.zekromaster.minecraft.terminal.capabilities;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import net.minecraft.class_206;
import net.minecraft.class_57;
import net.modificationstation.stationapi.api.util.API;
import net.modificationstation.stationapi.api.util.Identifier;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;

/**
 * Flexible access to an object of type {@code T} as long as it's attached to an {@link class_57}.
 *
 * @param <T> The underlying type
 * @param <CTX> A context object
 */
public final class EntityCapability<T, CTX> {

    final Multimap<String, EntityCapabilityHandler<T, CTX>> handlers = ArrayListMultimap.create();
    final ArrayList<EntityCapabilityHandler<T, CTX>> fallbacks = new ArrayList<>();

    public final Identifier identifier;
    public final Class<T> clazz;
    public final Class<CTX> ctxClass;

    private EntityCapability(
        Identifier identifier,
        Class<T> clazz,
        Class<CTX> ctxClass
    ) {
        this.identifier = identifier;
        this.clazz = clazz;
        this.ctxClass = ctxClass;
    }

    /**
     * @param identifier A unique identifier
     * @param clazz The underlying type
     * @param ctxClass A context type
     * @return A new entity capability with the given parameters
     * @param <T> The underlying type
     * @param <CTX> A context type
     */
    @API
    public static <T, CTX> EntityCapability<T, CTX> create(Identifier identifier, Class<T> clazz, Class<CTX> ctxClass) {
        return new EntityCapability<>(identifier, clazz, ctxClass);
    }

    /**
     * Same as {@link EntityCapability#create(Identifier, Class, Class)}, but the context object is always a
     * {@link Void} and thus is always null
     */
    @API
    public static <T> EntityCapability<T, Void> createVoid(Identifier identifier, Class<T> clazz) {
        return new EntityCapability<>(identifier, clazz, Void.class);
    }

    /**
     * @param entity The entity
     * @param ctx The context object
     * @return An instance of type {@code T}, or null if no handler provides one
     */
    @API
    public @Nullable T get(class_57 entity, CTX ctx) {
        for (var handler: this.handlers.get(class_206.method_734(entity))) {
            var value = handler.get(entity, ctx);
            if (value != null) {
                return value;
            }
        }
        for (var handler: this.fallbacks) {
            var value = handler.get(entity, ctx);
            if (value != null) {
                return value;
            }
        }
        return null;
    }


    @FunctionalInterface
    public interface EntityCapabilityHandler<T, CTX> {
        T get(class_57 entity, CTX context);
    }

}
